//---------------------------------------------------------------------------
// Copyright (c) 2016-2017 Embarcadero Technologies, Inc. All rights reserved.
//
// This software is the copyrighted property of Embarcadero Technologies, Inc.
// ("Embarcadero") and its licensors. You may only use this software if you
// are an authorized licensee of Delphi, C++Builder or RAD Studio
// (the "Embarcadero Products").  This software is subject to Embarcadero's
// standard software license and support agreement that accompanied your
// purchase of the Embarcadero Products and is considered a Redistributable,
// as such term is defined thereunder. Your use of this software constitutes
// your acknowledgement of your agreement to the foregoing software license
// and support agreement.
//---------------------------------------------------------------------------
unit FDownloadDemo;

interface

uses
  System.SysUtils, System.Types, System.UITypes, System.Classes, System.Variants,
  FMX.Types, FMX.Controls, FMX.Forms, FMX.Graphics, FMX.Dialogs, System.Net.URLClient, System.Net.HttpClient,
  System.Net.HttpClientComponent, FMX.StdCtrls, FMX.Edit, FMX.Controls.Presentation, FMX.ScrollBox, FMX.Memo,
  System.ImageList, FMX.ImgList, FMX.Memo.Types;

type
  TFormDownloadDemo = class(TForm)
    PanelTop: TPanel;
    PanelCenter: TPanel;
    LabelFile: TLabel;
    EditFileName: TEdit;
    BStartDownload: TButton;
    Memo1: TMemo;
    ImageList1: TImageList;
    LabelURL: TLabel;
    EditURL: TEdit;
    LabelGlobalSpeed: TLabel;
    ProgressBarDownload: TProgressBar;
    BStopDownload: TButton;
    procedure BStartDownloadClick(Sender: TObject);
    procedure ButtonCancelClick(Sender: TObject);
    procedure ReceiveDataEvent(const Sender: TObject; AContentLength: Int64; AReadCount: Int64; var Abort: Boolean);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
    FClient: THTTPClient;
    FGlobalStart: Cardinal;
    FAsyncResult: IAsyncResult;
    FDownloadStream: TStream;

    procedure DoEndDownload(const AsyncResult: IAsyncResult);
  public
    { Public declarations }
    procedure SampleDownload;
  end;

var
  FormDownloadDemo: TFormDownloadDemo;

implementation

{$R *.fmx}

uses
  System.IOUtils;

procedure TFormDownloadDemo.FormCreate(Sender: TObject);
begin
  FClient := THTTPClient.Create;
  FClient.OnReceiveData := ReceiveDataEvent;
  FClient.SecureProtocols := [THTTPSecureProtocol.TLS1, THTTPSecureProtocol.TLS11, THTTPSecureProtocol.TLS12];
end;

procedure TFormDownloadDemo.FormDestroy(Sender: TObject);
begin
  FDownloadStream.Free;
  FClient.Free;
end;

procedure TFormDownloadDemo.BStartDownloadClick(Sender: TObject);
begin
  BStartDownload.Enabled := False;
  SampleDownload;
end;

procedure TFormDownloadDemo.ButtonCancelClick(Sender: TObject);
begin
  (Sender as TButton).Enabled := False;
  FAsyncResult.Cancel;
end;

procedure TFormDownloadDemo.SampleDownload;
var
  URL: string;
  LResponse: IHTTPResponse;
  LFileName: string;
  LSize: Int64;
begin
  LFileName := TPath.Combine(TPath.GetDocumentsPath, EditFileName.Text);
  try
    URL := EditURL.Text;

    LResponse := FClient.Head(URL);
    LSize := LResponse.ContentLength;
    Memo1.Lines.Add(Format('Head response: %d - %s', [LResponse.StatusCode, LResponse.StatusText]));
    LResponse := nil;

    ProgressBarDownload.Max := LSize;
    ProgressBarDownload.Min := 0;
    ProgressBarDownload.Value := 0;
    LabelGlobalSpeed.Text := 'Global speed: 0 KB/s';

    Memo1.Lines.Add(Format('Downloading: "%s" (%d Bytes) into "%s"' , [EditFileName.Text, LSize, LFileName]));

    // Create the file that is going to be dowloaded
    FDownloadStream := TFileStream.Create(LFileName, fmCreate);
    FDownloadStream.Position := 0;

    // Start the download process
    FGlobalStart := TThread.GetTickCount;
    FAsyncResult := FClient.BeginGet(DoEndDownload, URL, FDownloadStream);

  finally
    BStopDownload.Enabled := FAsyncResult <> nil;
    BStartDownload.Enabled := FAsyncResult = nil;
  end;
end;

procedure TFormDownloadDemo.ReceiveDataEvent(const Sender: TObject; AContentLength, AReadCount: Int64;
  var Abort: Boolean);
var
  LTime: Cardinal;
  LSpeed: Integer;
begin
  LTime := TThread.GetTickCount - FGlobalStart;
  if LTime = 0 then
    Exit;
  LSpeed := (AReadCount * 1000) div LTime;
  TThread.Queue(nil,
    procedure
    begin
      ProgressBarDownload.Value := AReadCount;
      LabelGlobalSpeed.Text := Format('Global speed: %d KB/s', [LSpeed div 1024]);
    end);
end;

procedure TFormDownloadDemo.DoEndDownload(const AsyncResult: IAsyncResult);
var
  LAsyncResponse: IHTTPResponse;
begin
  try
    LAsyncResponse := THTTPClient.EndAsyncHTTP(AsyncResult);
    TThread.Synchronize(nil,
      procedure
      begin
        if AsyncResult.IsCancelled then
          Memo1.Lines.Add('Download Canceled')
        else
        begin
          Memo1.Lines.Add('Download Finished!');
          Memo1.Lines.Add(Format('Status: %d - %s', [LAsyncResponse.StatusCode, LAsyncResponse.StatusText]));
        end;

        BStopDownload.Enabled := False;
        BStartDownload.Enabled := True;
      end);
  finally
    LAsyncResponse := nil;
    FreeandNil(FDownloadStream);
  end;
end;

end.
